<?php
/**
 * This file includes the class Daln_Admin.
 *
 * @package live-news
 */

/**
 * This class should be used to work with the administrative side of WordPress.
 */
class Daln_Admin {

	/**
	 * Instance of the singleton class.
	 *
	 * @var null
	 */
	protected static $instance = null;

	/**
	 * Store an instance of the shared class.
	 *
	 * @var Daln_Shared|null
	 */
	private $shared = null;

	/**
	 * Screen id for the tickers page.
	 *
	 * @var null
	 */
	private $screen_id_tickers = null;

	/**
	 * Screen id for the featured page.
	 *
	 * @var null
	 */
	private $screen_id_featured = null;

	/**
	 * Screen id for the sliding page.
	 *
	 * @var null
	 */
	private $screen_id_sliding = null;

	/**
	 * Screen id for the import page.
	 *
	 * @var null
	 */
	private $screen_id_import = null;

	/**
	 * Screen id for the export page.
	 *
	 * @var null
	 */
	private $screen_id_export = null;

	/**
	 * Screen id for the maintenance page.
	 *
	 * @var null
	 */
	private $screen_id_maintenance = null;

	/**
	 * Screen id for the help page.
	 *
	 * @var null
	 */
	private $screen_id_help = null;

	/**
	 * Screen id for the options page.
	 *
	 * @var null
	 */
	private $screen_id_options = null;

	/**
	 * Construct.
	 */
	private function __construct() {

		// Assign an instance of the shared class.
		$this->shared = Daln_Shared::get_instance();

		// Load admin stylesheets and JavaScript.
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_styles' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );

		// Write in back-end head.
		add_action( 'admin_head', array( $this, 'wr_admin_head' ) );

		// Add the admin menu.
		add_action( 'admin_menu', array( $this, 'me_add_admin_menu' ) );

		// Load the options API registrations and callbacks.
		add_action( 'admin_init', array( $this, 'op_register_options' ) );

		// This hook is triggered during the creation of a new blog.
		add_action( 'wpmu_new_blog', array( $this, 'new_blog_create_options_and_tables' ), 10, 6 );

		// This hook is triggered during the deletion of a blog.
		add_action( 'delete_blog', array( $this, 'delete_blog_delete_options_and_tables' ), 10, 1 );

		// Export XML controller.
		add_action( 'init', array( $this, 'export_xml_controller' ) );
	}

	/**
	 * Return an instance of this class.
	 *
	 * @return self|null
	 */
	public static function get_instance() {

		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Write in the admin head.
	 *
	 * @return void
	 */
	public function wr_admin_head() {

		echo '<script type="text/javascript">';
		echo 'var daln_ajax_url = "' . admin_url( 'admin-ajax.php' ) . '";';
		echo 'var daln_nonce = "' . wp_create_nonce( 'live-news' ) . '";';
		echo 'var daln_admin_url ="' . get_admin_url() . '";';
		echo '</script>';
	}

	public function enqueue_admin_styles() {

		$screen = get_current_screen();

		// Menu tickers.
		if ( $screen->id === $this->screen_id_tickers ) {
			wp_enqueue_style( 'wp-color-picker' );
			wp_enqueue_style( $this->shared->get( 'slug' ) . '-menu-sliding', $this->shared->get( 'url' ) . 'admin/assets/css/menu-tickers.css', array(), $this->shared->get( 'ver' ) );
			wp_enqueue_style( $this->shared->get( 'slug' ) . '-framework-menu', $this->shared->get( 'url' ) . 'admin/assets/css/framework/menu.css', array(), $this->shared->get( 'ver' ) );
			wp_enqueue_style( $this->shared->get( 'slug' ) . '-jquery-ui-tooltip', $this->shared->get( 'url' ) . 'admin/assets/css/jquery-ui-tooltip.css', array(), $this->shared->get( 'ver' ) );
			wp_enqueue_style( $this->shared->get( 'slug' ) . '-select2', $this->shared->get( 'url' ) . 'admin/assets/inc/select2/dist/css/select2.min.css', array(), $this->shared->get( 'ver' ) );
			wp_enqueue_style( $this->shared->get( 'slug' ) . '-select2-custom', $this->shared->get( 'url' ) . 'admin/assets/css/select2-custom.css', array(), $this->shared->get( 'ver' ) );
		}

		// Menu featured.
		if ( $screen->id === $this->screen_id_featured ) {
			wp_enqueue_style( $this->shared->get( 'slug' ) . '-menu-featured', $this->shared->get( 'url' ) . 'admin/assets/css/menu-featured.css', array(), $this->shared->get( 'ver' ) );
			wp_enqueue_style( $this->shared->get( 'slug' ) . '-framework-menu', $this->shared->get( 'url' ) . 'admin/assets/css/framework/menu.css', array(), $this->shared->get( 'ver' ) );
			wp_enqueue_style( $this->shared->get( 'slug' ) . '-jquery-ui-tooltip', $this->shared->get( 'url' ) . 'admin/assets/css/jquery-ui-tooltip.css', array(), $this->shared->get( 'ver' ) );
			wp_enqueue_style( $this->shared->get( 'slug' ) . '-select2', $this->shared->get( 'url' ) . 'admin/assets/inc/select2/dist/css/select2.min.css', array(), $this->shared->get( 'ver' ) );
			wp_enqueue_style( $this->shared->get( 'slug' ) . '-select2-custom', $this->shared->get( 'url' ) . 'admin/assets/css/select2-custom.css', array(), $this->shared->get( 'ver' ) );
		}

		// Menu sliding.
		if ( $screen->id === $this->screen_id_sliding ) {
			wp_enqueue_style( 'wp-color-picker' );
			wp_enqueue_style( $this->shared->get( 'slug' ) . '-menu-sliding', $this->shared->get( 'url' ) . 'admin/assets/css/menu-sliding.css', array(), $this->shared->get( 'ver' ) );
			wp_enqueue_style( $this->shared->get( 'slug' ) . '-framework-menu', $this->shared->get( 'url' ) . 'admin/assets/css/framework/menu.css', array(), $this->shared->get( 'ver' ) );
			wp_enqueue_style( $this->shared->get( 'slug' ) . '-jquery-ui-tooltip', $this->shared->get( 'url' ) . 'admin/assets/css/jquery-ui-tooltip.css', array(), $this->shared->get( 'ver' ) );
			wp_enqueue_style( $this->shared->get( 'slug' ) . '-select2', $this->shared->get( 'url' ) . 'admin/assets/inc/select2/dist/css/select2.min.css', array(), $this->shared->get( 'ver' ) );
			wp_enqueue_style( $this->shared->get( 'slug' ) . '-select2-custom', $this->shared->get( 'url' ) . 'admin/assets/css/select2-custom.css', array(), $this->shared->get( 'ver' ) );
		}

		// Menu maintenance.
		if ( $screen->id === $this->screen_id_maintenance ) {

			wp_enqueue_style(
				$this->shared->get( 'slug' ) . '-framework-menu',
				$this->shared->get( 'url' ) . 'admin/assets/css/framework/menu.css',
				array(),
				$this->shared->get( 'ver' )
			);

			// jQuery UI Dialog.
			wp_enqueue_style(
				$this->shared->get( 'slug' ) . '-jquery-ui-dialog',
				$this->shared->get( 'url' ) . 'admin/assets/css/jquery-ui-dialog.css',
				array(),
				$this->shared->get( 'ver' )
			);
			wp_enqueue_style(
				$this->shared->get( 'slug' ) . '-jquery-ui-dialog-custom',
				$this->shared->get( 'url' ) . 'admin/assets/css/jquery-ui-dialog-custom.css',
				array(),
				$this->shared->get( 'ver' )
			);

			// jQuery UI Tooltip.
			wp_enqueue_style( $this->shared->get( 'slug' ) . '-jquery-ui-tooltip', $this->shared->get( 'url' ) . 'admin/assets/css/jquery-ui-tooltip.css', array(), $this->shared->get( 'ver' ) );

			// Select2.
			wp_enqueue_style( $this->shared->get( 'slug' ) . '-select2', $this->shared->get( 'url' ) . 'admin/assets/inc/select2/dist/css/select2.min.css', array(), $this->shared->get( 'ver' ) );
			wp_enqueue_style( $this->shared->get( 'slug' ) . '-select2-custom', $this->shared->get( 'url' ) . 'admin/assets/css/select2-custom.css', array(), $this->shared->get( 'ver' ) );

		}

		// Menu help.
		if ( $screen->id === $this->screen_id_help ) {
			wp_enqueue_style(
				$this->shared->get( 'slug' ) . '-menu-help',
				$this->shared->get( 'url' ) . 'admin/assets/css/menu-help.css',
				array(),
				$this->shared->get( 'ver' )
			);
		}

		// Menu options.
		if ( $screen->id === $this->screen_id_options ) {
			wp_enqueue_style( $this->shared->get( 'slug' ) . '-framework-options', $this->shared->get( 'url' ) . 'admin/assets/css/framework/options.css', array(), $this->shared->get( 'ver' ) );
			wp_enqueue_style( $this->shared->get( 'slug' ) . '-jquery-ui-tooltip', $this->shared->get( 'url' ) . 'admin/assets/css/jquery-ui-tooltip.css', array(), $this->shared->get( 'ver' ) );
			wp_enqueue_style( $this->shared->get( 'slug' ) . '-select2', $this->shared->get( 'url' ) . 'admin/assets/inc/select2/dist/css/select2.min.css', array(), $this->shared->get( 'ver' ) );
			wp_enqueue_style( $this->shared->get( 'slug' ) . '-select2-custom', $this->shared->get( 'url' ) . 'admin/assets/css/select2-custom.css', array(), $this->shared->get( 'ver' ) );
		}
	}

	/**
	 * Enqueue admin-specific javascript.
	 *
	 * @return void
	 */
	public function enqueue_admin_scripts() {

		$wp_localize_script_data = array(
			'deleteText' => esc_html__( 'Delete', 'daextlwcn' ),
			'cancelText' => esc_html__( 'Cancel', 'daextlwcn' ),
		);

		$screen = get_current_screen();

		// Menu tickers.
		if ( $screen->id === $this->screen_id_tickers ) {
			wp_enqueue_script( 'jquery-ui-tooltip' );
			wp_enqueue_script( $this->shared->get( 'slug' ) . '-menu-tickers', $this->shared->get( 'url' ) . 'admin/assets/js/menu-tickers.js', 'jquery', $this->shared->get( 'ver' ) );
			wp_enqueue_script( $this->shared->get( 'slug' ) . '-jquery-ui-tooltip-init', $this->shared->get( 'url' ) . 'admin/assets/js/jquery-ui-tooltip-init.js', 'jquery', $this->shared->get( 'ver' ) );
			wp_enqueue_script( $this->shared->get( 'slug' ) . '-wp-color-picker-init', $this->shared->get( 'url' ) . 'admin/assets/js/wp-color-picker-init.js', array( 'wp-color-picker' ), false, true );
			wp_enqueue_script( $this->shared->get( 'slug' ) . '-select2', $this->shared->get( 'url' ) . 'admin/assets/inc/select2/dist/js/select2.full.min.js', 'jquery', $this->shared->get( 'ver' ) );
			wp_enqueue_script( $this->shared->get( 'slug' ) . '-jquery-ui-select2-init-tickers', $this->shared->get( 'url' ) . 'admin/assets/js/select2-init-tickers.js', 'jquery', $this->shared->get( 'ver' ) );
			wp_enqueue_media();
			wp_enqueue_script( $this->shared->get( 'slug' ) . '-media-uploader', $this->shared->get( 'url' ) . 'admin/assets/js/media-uploader.js', 'jquery', $this->shared->get( 'ver' ) );
		}

		// Menu featured.
		if ( $screen->id === $this->screen_id_featured ) {
			wp_enqueue_script( 'jquery-ui-tooltip' );
			wp_enqueue_script( $this->shared->get( 'slug' ) . '-jquery-ui-tooltip-init', $this->shared->get( 'url' ) . 'admin/assets/js/jquery-ui-tooltip-init.js', 'jquery', $this->shared->get( 'ver' ) );
			wp_enqueue_script( $this->shared->get( 'slug' ) . '-select2', $this->shared->get( 'url' ) . 'admin/assets/inc/select2/dist/js/select2.min.js', 'jquery', $this->shared->get( 'ver' ) );
			wp_enqueue_script( $this->shared->get( 'slug' ) . '-jquery-ui-select2-init-featured', $this->shared->get( 'url' ) . 'admin/assets/js/select2-init-featured.js', 'jquery', $this->shared->get( 'ver' ) );
			wp_enqueue_script( $this->shared->get( 'slug' ) . '-menu-featured', $this->shared->get( 'url' ) . 'admin/assets/js/menu-featured.js', 'jquery', $this->shared->get( 'ver' ) );
		}

		// Menu sliding.
		if ( $screen->id === $this->screen_id_sliding ) {
			wp_enqueue_script( 'jquery-ui-tooltip' );
			wp_enqueue_script( $this->shared->get( 'slug' ) . '-jquery-ui-tooltip-init', $this->shared->get( 'url' ) . 'admin/assets/js/jquery-ui-tooltip-init.js', 'jquery', $this->shared->get( 'ver' ) );
			wp_enqueue_script( $this->shared->get( 'slug' ) . '-select2', $this->shared->get( 'url' ) . 'admin/assets/inc/select2/dist/js/select2.min.js', 'jquery', $this->shared->get( 'ver' ) );
			wp_enqueue_script( $this->shared->get( 'slug' ) . '-jquery-ui-select2-init-sliding', $this->shared->get( 'url' ) . 'admin/assets/js/select2-init-sliding.js', 'jquery', $this->shared->get( 'ver' ) );
			wp_enqueue_script( $this->shared->get( 'slug' ) . '-wp-color-picker-init', $this->shared->get( 'url' ) . 'admin/assets/js/wp-color-picker-init.js', array( 'wp-color-picker' ), false, true );
			wp_enqueue_script( $this->shared->get( 'slug' ) . '-menu-sliding', $this->shared->get( 'url' ) . 'admin/assets/js/menu-sliding.js', 'jquery', $this->shared->get( 'ver' ) );
			wp_enqueue_media();
			wp_enqueue_script( $this->shared->get( 'slug' ) . '-media-uploader', $this->shared->get( 'url' ) . 'admin/assets/js/media-uploader.js', 'jquery', $this->shared->get( 'ver' ) );
		}

		// Menu Maintenance.
		if ( $screen->id === $this->screen_id_maintenance ) {

			// Select2.
			wp_enqueue_script( $this->shared->get( 'slug' ) . '-select2', $this->shared->get( 'url' ) . 'admin/assets/inc/select2/dist/js/select2.min.js', 'jquery', $this->shared->get( 'ver' ) );

			// Maintenance Menu.
			wp_enqueue_script(
				$this->shared->get( 'slug' ) . '-menu-maintenance',
				$this->shared->get( 'url' ) . 'admin/assets/js/menu-maintenance.js',
				array( 'jquery', 'jquery-ui-dialog', $this->shared->get( 'slug' ) . '-select2' ),
				$this->shared->get( 'ver' )
			);
			wp_localize_script(
				$this->shared->get( 'slug' ) . '-menu-maintenance',
				'objectL10n',
				$wp_localize_script_data
			);

			// jQuery UI Tooltip.
			wp_enqueue_script( 'jquery-ui-tooltip' );
			wp_enqueue_script(
				$this->shared->get( 'slug' ) . '-jquery-ui-tooltip-init',
				$this->shared->get( 'url' ) . 'admin/assets/js/jquery-ui-tooltip-init.js',
				'jquery',
				$this->shared->get( 'ver' )
			);

		}

		// Menu options.
		if ( $screen->id == $this->screen_id_options ) {
			wp_enqueue_script( 'jquery-ui-tooltip' );
			wp_enqueue_script( $this->shared->get( 'slug' ) . '-jquery-ui-tooltip-init', $this->shared->get( 'url' ) . 'admin/assets/js/jquery-ui-tooltip-init.js', 'jquery', $this->shared->get( 'ver' ) );
			wp_enqueue_script( $this->shared->get( 'slug' ) . '-select2', $this->shared->get( 'url' ) . 'admin/assets/inc/select2/dist/js/select2.min.js', 'jquery', $this->shared->get( 'ver' ) );
			wp_enqueue_script( $this->shared->get( 'slug' ) . '-select2-init-options', $this->shared->get( 'url' ) . 'admin/assets/js/select2-init-options.js', 'jquery', $this->shared->get( 'ver' ) );
		}
	}

	/**
	 * Plugin activation.
	 *
	 * @param $networkwide
	 *
	 * @return void
	 */
	public function ac_activate( $networkwide ) {

		/*
		 * Create options and tables for all the sites in the network.
		 */
		if ( function_exists( 'is_multisite' ) && is_multisite() ) {

			/*
			 * if this is a "Network Activation" create the options and tables
			 * for each blog
			 */
			if ( $networkwide ) {

				// Get the current blog id.
				global $wpdb;
				$current_blog = $wpdb->blogid;

				// Create an array with all the blog ids.
				$blogids = $wpdb->get_col( "SELECT blog_id FROM $wpdb->blogs" );

				// Iterate through all the blogs.
				foreach ( $blogids as $blog_id ) {

					// Switch to the iterated blog.
					switch_to_blog( $blog_id );

					// Create options and tables for the iterated blog.
					$this->ac_initialize_options();
					$this->ac_create_database_tables();

				}

				// Switch to the current blog.
				switch_to_blog( $current_blog );

			} else {
				/*
				 * If this is not a "Network Activation" create options and
				 * tables only for the current blog.
				 */
				$this->ac_initialize_options();
				$this->ac_create_database_tables();

			}
		} else {
			/*
			 * If this is not a multisite installation create options and
			 * tables only for the current blog.
			 */
			$this->ac_initialize_options();
			$this->ac_create_database_tables();

		}
	}

	/**
	 * Create the options and tables for the newly created blog.
	 *
	 * @param $blog_id
	 * @param $user_id
	 * @param $domain
	 * @param $path
	 * @param $site_id
	 * @param $meta
	 *
	 * @return void
	 */
	public function new_blog_create_options_and_tables( $blog_id, $user_id, $domain, $path, $site_id, $meta ) {

		global $wpdb;

		/*
		 * if the plugin is "Network Active" create the options and tables for
		 * this new blog
		 */
		if ( is_plugin_active_for_network( 'uberchart/init.php' ) ) {

			// Get the id of the current blog.
			$current_blog = $wpdb->blogid;

			// Switch to the blog that is being activated.
			switch_to_blog( $blog_id );

			// Create options and database tables for the new blog.
			$this->ac_initialize_options();
			$this->ac_create_database_tables();

			// Switch to the current blog.
			switch_to_blog( $current_blog );

		}
	}

	/**
	 * Delete options and tables for the deleted blog.
	 *
	 * @param $blog_id
	 *
	 * @return void
	 */
	public function delete_blog_delete_options_and_tables( $blog_id ) {

		global $wpdb;

		// Get the id of the current blog.
		$current_blog = $wpdb->blogid;

		// Switch to the blog that is being activated.
		switch_to_blog( $blog_id );

		// Create options and database tables for the new blog.
		$this->un_delete_options();
		$this->un_delete_database_tables();

		// Switch to the current blog.
		switch_to_blog( $current_blog );
	}

	/**
	 * Initialize plugin options.
	 *
	 * @return void
	 */
	private function ac_initialize_options() {

		foreach ( $this->shared->get( 'options' ) as $key => $value ) {
			add_option( $key, $value );
		}
	}

	/**
	 * Create the plugin database tables.
	 *
	 * @return void
	 */
	private function ac_create_database_tables() {

		global $wpdb;

		// Get the database character collate that will be appended at the end of each query.
		$charset_collate = $wpdb->get_charset_collate();

		// Check database version and create the database.
		if ( intval( get_option( $this->shared->get( 'slug' ) . '_database_version' ), 10 ) < 3 ) {

			require_once ABSPATH . 'wp-admin/includes/upgrade.php';

			// Create *prefix*_daln_tickers.
			global $wpdb;
			$table_name = $wpdb->prefix . $this->shared->get( 'slug' ) . '_tickers';
			$sql        = "CREATE TABLE $table_name (
                  `name` varchar(100) NOT NULL DEFAULT '',
                  `id` bigint(20) NOT NULL AUTO_INCREMENT PRIMARY KEY,
                  `target` int(11) NOT NULL DEFAULT '1',
                  `url` TEXT NOT NULL DEFAULT '',
                  `source` int(11) NOT NULL DEFAULT '1',
                  `url_rss` varchar(2083) NOT NULL DEFAULT '',
                  `category` bigint(20) NOT NULL DEFAULT '0',
                  `twitter_username` varchar(255) NOT NULL DEFAULT '',
                  `twitter_oauth_access_token` varchar(255) NOT NULL DEFAULT '',
                  `twitter_oauth_access_token_secret` varchar(255) NOT NULL DEFAULT '',
                  `twitter_consumer_key` varchar(255) NOT NULL DEFAULT '',
                  `twitter_consumer_secret` varchar(255) NOT NULL DEFAULT '',
                  `twitter_strip_links` tinyint(1) DEFAULT '0',
                  `twitter_strip_hashtags` tinyint(1) DEFAULT '0',
                  `twitter_strip_usernames` tinyint(1) DEFAULT '0',
                  `open_links_new_tab` tinyint(1) DEFAULT '0',
                  `clock_offset` int(11) NOT NULL DEFAULT '0',
                  `clock_format` varchar(40) NOT NULL DEFAULT 'HH:mm',
                  `clock_source` int(11) NOT NULL DEFAULT '2',
                  `clock_autoupdate` tinyint(1) DEFAULT '1',
                  `clock_autoupdate_time` int(11) NOT NULL DEFAULT '10',
                  `sliding_speed` int(11) NOT NULL DEFAULT '5',
                  `sliding_delay` int(11) NOT NULL DEFAULT '0',
                  `number_of_sliding_news` int(11) NOT NULL DEFAULT '10',
                  `featured_title_maximum_length` int(11) NOT NULL DEFAULT '255',
                  `featured_excerpt_maximum_length` int(11) NOT NULL DEFAULT '255',
                  `sliding_news_maximum_length` int(11) NOT NULL DEFAULT '255',
                  `open_news_as_default` tinyint(1) DEFAULT '1',
                  `hide_featured_news` int(11) NOT NULL DEFAULT '1',
                  `hide_clock` tinyint(1) DEFAULT '0',
                  `enable_rtl_layout` tinyint(1) DEFAULT '0',
                  `cached_cycles` int(11) NOT NULL DEFAULT '0',
                  `featured_news_background_color` varchar(7) DEFAULT NULL,
                  `sliding_news_background_color` varchar(7) DEFAULT NULL,
                  `sliding_news_background_color_opacity` float DEFAULT NULL,
                  `font_family` varchar(255) DEFAULT NULL,
                  `google_font` varchar(255) DEFAULT NULL,
                  `featured_title_font_size` int(11) NOT NULL DEFAULT '38',
                  `featured_excerpt_font_size` int(11) NOT NULL DEFAULT '28',
                  `sliding_news_font_size` int(11) NOT NULL DEFAULT '28',
                  `clock_font_size` int(11) NOT NULL DEFAULT '28',
                  `enable_with_mobile_devices` tinyint(1) DEFAULT '0',
                  `open_button_image` varchar(2083) NOT NULL DEFAULT '',
                  `close_button_image` varchar(2083) NOT NULL DEFAULT '',
                  `clock_background_image` varchar(2083) NOT NULL DEFAULT '',
                  `featured_news_title_color` varchar(7) DEFAULT NULL,
                  `featured_news_title_color_hover` varchar(7) DEFAULT NULL,
                  `featured_news_excerpt_color` varchar(7) DEFAULT NULL,
                  `sliding_news_color` varchar(7) DEFAULT NULL,
                  `sliding_news_color_hover` varchar(7) DEFAULT NULL,
                  `clock_text_color` varchar(7) DEFAULT NULL,
                  `featured_news_background_color_opacity` float DEFAULT NULL,
                  `enable_ticker` tinyint(1) DEFAULT '1',
                  `enable_links` tinyint(1) DEFAULT '1',
                  `transient_expiration` int(11) NOT NULL DEFAULT '0',
                  `sliding_news_margin` int(11) NOT NULL DEFAULT '84',
                  `sliding_news_padding` int(11) NOT NULL DEFAULT '28',
                  `url_mode` tinyint(1) DEFAULT '0'
            ) $charset_collate";

			dbDelta( $sql );

			// Create *prefix*_daln_featured_news.
			global $wpdb;
			$table_name = $wpdb->prefix . $this->shared->get( 'slug' ) . '_featured_news';
			$sql        = "CREATE TABLE $table_name (
                  `id` int(11) NOT NULL AUTO_INCREMENT PRIMARY KEY,
                  `news_title` varchar(1000) NOT NULL DEFAULT '',
                  `news_excerpt` varchar(1000) NOT NULL DEFAULT '',
                  `url` varchar(2083) NOT NULL DEFAULT '',
                  `ticker_id` bigint(20) NOT NULL
            ) $charset_collate";

			dbDelta( $sql );

			// Create *prefix*_daln_sliding_news.
			global $wpdb;
			$table_name = $wpdb->prefix . $this->shared->get( 'slug' ) . '_sliding_news';
			$sql        = "CREATE TABLE $table_name (
                  `id` int(11) NOT NULL AUTO_INCREMENT PRIMARY KEY,
                  `news_title` varchar(1000) NOT NULL DEFAULT '',
                  `url` varchar(2083) NOT NULL DEFAULT '',
                  `ticker_id` bigint(20) NOT NULL,
                  `text_color` varchar(7) DEFAULT NULL,
                  `text_color_hover` varchar(7) DEFAULT NULL,
                  `background_color` varchar(7) DEFAULT NULL,
                  `background_color_opacity` float DEFAULT NULL,
                  `image_before` varchar(2083) NOT NULL DEFAULT '',
                  `image_after` varchar(2083) NOT NULL DEFAULT ''
            ) $charset_collate";

			dbDelta( $sql );

			// Update database version.
			update_option( $this->shared->get( 'slug' ) . '_database_version', '2' );

		}
	}

	/**
	 * Plugin delete.
	 *
	 * @return void
	 */
	public static function un_delete() {
		/*
		 * Delete options and tables for all the sites in the network.
		 */
		if ( function_exists( 'is_multisite' ) && is_multisite() ) {

			// Get the current blog id.
			global $wpdb;
			$current_blog = $wpdb->blogid;

			// Create an array with all the blog ids.
			$blogids = $wpdb->get_col( "SELECT blog_id FROM $wpdb->blogs" );

			// Iterate through all the blogs.
			foreach ( $blogids as $blog_id ) {

				// Switch to the iterated blog.
				switch_to_blog( $blog_id );

				// Create options and tables for the iterated blog.
				self::un_delete_options();
				self::un_delete_database_tables();

			}

			// Switch to the current blog.
			switch_to_blog( $current_blog );

		} else {
			/*
			 * if this is not a multisite installation delete options and
			 * tables only for the current blog
			 */
			self::un_delete_options();
			self::un_delete_database_tables();

		}
	}

	/**
	 * Delete plugin options.
	 *
	 * @return void
	 */
	public static function un_delete_options() {

		// Assign an instance of Daln_Shared.
		$shared = Daln_Shared::get_instance();

		foreach ( $shared->get( 'options' ) as $key => $value ) {
			delete_option( $key );
		}
	}

	/**
	 * Delete plugin database tables.
	 *
	 * @return void
	 */
	public static function un_delete_database_tables() {

		// Assign an instance of Daln_Shared.
		$shared = Daln_Shared::get_instance();

		global $wpdb;

		// Delete transients associated with the table prefix '_tickers'.
		$table_name = $wpdb->prefix . $shared->get( 'slug' ) . '_tickers';
		$results    = $wpdb->get_results( "SELECT id FROM $table_name", ARRAY_A );
		foreach ( $results as $result ) {
			delete_transient( 'daln_ticker_' . $result['id'] );
		}

		// Delete table prefix + '_tickers'.
		$table_name = $wpdb->prefix . $shared->get( 'slug' ) . '_tickers';
		$sql        = "DROP TABLE $table_name";
		$wpdb->query( $sql );

		// Delete table prefix + '_featured_news'.
		$table_name = $wpdb->prefix . $shared->get( 'slug' ) . '_featured_news';
		$sql        = "DROP TABLE $table_name";
		$wpdb->query( $sql );

		// Delete table prefix + '_sliding_news'.
		$table_name = $wpdb->prefix . $shared->get( 'slug' ) . '_sliding_news';
		$sql        = "DROP TABLE $table_name";
		$wpdb->query( $sql );
	}

	/**
	 * Register the admin menu.
	 *
	 * @return void
	 */
	public function me_add_admin_menu() {

		add_menu_page(
			'LN',
			esc_attr__( 'Live News', 'daln' ),
			get_option( $this->shared->get( 'slug' ) . '_tickers_menu_capability' ),
			$this->shared->get( 'slug' ) . '-tickers',
			array( $this, 'me_display_menu_tickers' ),
			'dashicons-admin-site'
		);

		$this->screen_id_tickers = add_submenu_page(
			$this->shared->get( 'slug' ) . '-tickers',
			esc_attr__( 'News Tickers', 'daln' ),
			esc_attr__( 'News Tickers', 'daln' ),
			get_option( $this->shared->get( 'slug' ) . '_tickers_menu_capability' ),
			$this->shared->get( 'slug' ) . '-tickers',
			array( $this, 'me_display_menu_tickers' )
		);

		$this->screen_id_featured = add_submenu_page(
			$this->shared->get( 'slug' ) . '-tickers',
			esc_attr__( 'Featured News', 'daln' ),
			esc_attr__( 'Featured News', 'daln' ),
			get_option( $this->shared->get( 'slug' ) . '_featured_menu_capability' ),
			$this->shared->get( 'slug' ) . '-featured',
			array( $this, 'me_display_menu_featured' )
		);

		$this->screen_id_sliding = add_submenu_page(
			$this->shared->get( 'slug' ) . '-tickers',
			esc_attr__( 'Sliding News', 'daln' ),
			esc_attr__( 'Sliding News', 'daln' ),
			get_option( $this->shared->get( 'slug' ) . '_sliding_menu_capability' ),
			$this->shared->get( 'slug' ) . '-sliding',
			array( $this, 'me_display_menu_sliding' )
		);

		$this->screen_id_import = add_submenu_page(
			$this->shared->get( 'slug' ) . '-tickers',
			esc_attr__( 'Import', 'daln' ),
			esc_attr__( 'Import', 'daln' ),
			get_option( $this->shared->get( 'slug' ) . '_import_menu_capability' ),
			$this->shared->get( 'slug' ) . '-import',
			array( $this, 'me_display_menu_import' )
		);

		$this->screen_id_export = add_submenu_page(
			$this->shared->get( 'slug' ) . '-tickers',
			esc_attr__( 'Export', 'daln' ),
			esc_attr__( 'Export', 'daln' ),
			get_option( $this->shared->get( 'slug' ) . '_export_menu_capability' ),
			$this->shared->get( 'slug' ) . '-export',
			array( $this, 'me_display_menu_export' )
		);

		$this->screen_id_maintenance = add_submenu_page(
			$this->shared->get( 'slug' ) . '-tickers',
			esc_attr__( 'Maintenance', 'daln' ),
			esc_attr__( 'Maintenance', 'daln' ),
			get_option( $this->shared->get( 'slug' ) . '_maintenance_menu_capability' ),
			$this->shared->get( 'slug' ) . '-maintenance',
			array( $this, 'me_display_menu_maintenance' )
		);

		$this->screen_id_help = add_submenu_page(
			$this->shared->get( 'slug' ) . '-tickers',
			esc_attr__( 'Help', 'daln' ),
			esc_attr__( 'Help', 'daln' ),
			'manage_options',
			$this->shared->get( 'slug' ) . '-help',
			array( $this, 'me_display_menu_help' )
		);

		$this->screen_id_options = add_submenu_page(
			$this->shared->get( 'slug' ) . '-tickers',
			esc_attr__( 'Options', 'daln' ),
			esc_attr__( 'Options', 'daln' ),
			'manage_options',
			$this->shared->get( 'slug' ) . '-options',
			array( $this, 'me_display_menu_options' )
		);
	}

	/**
	 * Includes the tickers view.
	 *
	 * @return void
	 */
	public function me_display_menu_tickers() {
		include_once 'view/tickers.php';
	}

	/**
	 * Includes the featured view.
	 *
	 * @return void
	 */
	public function me_display_menu_featured() {
		include_once 'view/featured.php';
	}

	/**
	 * Includes the sliding view.
	 *
	 * @return void
	 */
	public function me_display_menu_sliding() {
		include_once 'view/sliding.php';
	}

	/**
	 * Includes the import view.
	 *
	 * @return void
	 */
	public function me_display_menu_import() {
		include_once 'view/import.php';
	}

	/**
	 * Includes the export view.
	 *
	 * @return void
	 */
	public function me_display_menu_export() {
		include_once 'view/export.php';
	}

	/**
	 * Includes the maintenance view.
	 *
	 * @return void
	 */
	public function me_display_menu_maintenance() {
		include_once 'view/maintenance.php';
	}

	/**
	 * Includes the help view.
	 *
	 * @return void
	 */
	public function me_display_menu_help() {
		include_once 'view/help.php';
	}

	/**
	 * Includes the options view.
	 *
	 * @return void
	 */
	public function me_display_menu_options() {
		include_once 'view/options.php';
	}

	/**
	 * Register options.
	 *
	 * @return void
	 */
	public function op_register_options() {

		// section general ----------------------------------------------------------
		add_settings_section(
			'daln_general_settings_section',
			null,
			null,
			'daln_general_options'
		);

		add_settings_field(
			'detect_url_mode',
			esc_attr__( 'Detect URL Mode', 'daln' ),
			array( $this, 'detect_url_mode_callback' ),
			'daln_general_options',
			'daln_general_settings_section'
		);

		register_setting(
			'daln_general_options',
			'daln_detect_url_mode',
			array( $this, 'detect_url_mode_validation' )
		);

		add_settings_field(
			'load_momentjs',
			esc_attr__( 'Load Moment.js', 'daln' ),
			array( $this, 'load_momentjs_callback' ),
			'daln_general_options',
			'daln_general_settings_section'
		);

		register_setting(
			'daln_general_options',
			'daln_load_momentjs',
			array( $this, 'load_momentjs_validation' )
		);

		add_settings_field(
			'tickers_menu_capability',
			esc_attr__( 'Tickers Menu Capability', 'daln' ),
			array( $this, 'tickers_menu_capability_callback' ),
			'daln_general_options',
			'daln_general_settings_section'
		);

		register_setting(
			'daln_general_options',
			'daln_tickers_menu_capability',
			'sanitize_key'
		);

		add_settings_field(
			'featured_menu_capability',
			esc_attr__( 'Featured News Menu Capability', 'daln' ),
			array( $this, 'featured_menu_capability_callback' ),
			'daln_general_options',
			'daln_general_settings_section'
		);

		register_setting(
			'daln_general_options',
			'daln_featured_menu_capability',
			'sanitize_key'
		);

		add_settings_field(
			'sliding_menu_capability',
			esc_attr__( 'Sliding News Menu Capability', 'daln' ),
			array( $this, 'sliding_menu_capability_callback' ),
			'daln_general_options',
			'daln_general_settings_section'
		);

		register_setting(
			'daln_general_options',
			'daln_sliding_menu_capability',
			'sanitize_key'
		);

		add_settings_field(
			'import_menu_capability',
			esc_attr__( 'Import Menu Capability', 'daln' ),
			array( $this, 'import_menu_capability_callback' ),
			'daln_general_options',
			'daln_general_settings_section'
		);

		register_setting(
			'daln_general_options',
			'daln_import_menu_capability',
			'sanitize_key'
		);

		add_settings_field(
			'export_menu_capability',
			esc_attr__( 'Export Menu Capability', 'daln' ),
			array( $this, 'export_menu_capability_callback' ),
			'daln_general_options',
			'daln_general_settings_section'
		);

		register_setting(
			'daln_general_options',
			'daln_export_menu_capability',
			'sanitize_key'
		);

		add_settings_field(
			'maintenance_menu_capability',
			esc_attr__( 'Maintenance Menu Capability', 'daln' ),
			array( $this, 'maintenance_menu_capability_callback' ),
			'daln_general_options',
			'daln_general_settings_section'
		);

		register_setting(
			'daln_general_options',
			'daln_maintenance_menu_capability',
			'sanitize_key'
		);
	}


	public function detect_url_mode_callback() {

		$html  = '<select id="daln-detect-url-mode" name="daln_detect_url_mode" class="daext-display-none">';
		$html .= '<option ' . selected( get_option( 'daln_detect_url_mode' ), 'server_variable', false ) . ' value="server_variable">' . esc_attr__( 'Server Variable', 'dahm' ) . '</option>';
		$html .= '<option ' . selected( get_option( 'daln_detect_url_mode' ), 'wp_request', false ) . ' value="wp_request">' . esc_attr__( 'WP Request', 'dahm' ) . '</option>';
		$html .= '</select>';
		$html .= '<div class="help-icon" title="' . esc_attr__( 'Select the method used to detect the URL of the page.', 'dahm' ) . '"></div>';

		echo $html;
	}

	/**
	 * Validate the detect_url_mode option.
	 *
	 * @param $input
	 *
	 * @return string
	 */
	public function detect_url_mode_validation( $input ) {

		if ( $input === 'server_variable' or $input === 'wp_request' ) {
			$output = $input;
		} else {
			$output = 'server_variable';
		}

		return $output;
	}

	public function load_momentjs_callback() {

		$html  = '<select id="daln-load-momentjs" name="daln_load_momentjs" class="daext-display-none">';
		$html .= '<option ' . selected( intval( get_option( 'daln_load_momentjs' ) ), 0, false ) . ' value="0">' . esc_attr__( 'No', 'daln' ) . '</option>';
		$html .= '<option ' . selected( intval( get_option( 'daln_load_momentjs' ) ), 1, false ) . ' value="1">' . esc_attr__( 'Yes', 'daln' ) . '</option>';
		$html .= '</select>';
		$html .= '<div class="help-icon" title="' . esc_attr__( 'Please select "No" only if other components of your website are already loading the Moment.js library.', 'daln' ) . '"></div>';

		echo $html;
	}

	public function load_momentjs_validation( $input ) {

		return intval( $input, 10 ) === 1 ? '1' : '0';
	}

	public function tickers_menu_capability_callback() {

		$html  = '<input autocomplete="off" type="text" id="daln-tickers-menu-capability" name="daln_tickers_menu_capability" class="regular-text" value="' . esc_attr( get_option( 'daln_tickers_menu_capability' ) ) . '" />';
		$html .= '<div class="help-icon" title="' . esc_attr__( 'The capability required to get access on the "News Tickers" menu.', 'daln' ) . '"></div>';

		echo $html;
	}

	public function featured_menu_capability_callback() {

		$html  = '<input autocomplete="off" type="text" id="daln-featured-menu-capability" name="daln_featured_menu_capability" class="regular-text" value="' . esc_attr( get_option( 'daln_featured_menu_capability' ) ) . '" />';
		$html .= '<div class="help-icon" title="' . esc_attr__( 'The capability required to get access on the "Featured News" menu.', 'daln' ) . '"></div>';

		echo $html;
	}

	public function sliding_menu_capability_callback() {

		$html  = '<input autocomplete="off" type="text" id="daln-sliding-menu-capability" name="daln_sliding_menu_capability" class="regular-text" value="' . esc_attr( get_option( 'daln_sliding_menu_capability' ) ) . '" />';
		$html .= '<div class="help-icon" title="' . esc_attr__( 'The capability required to get access on the "Sliding News" menu.', 'daln' ) . '"></div>';

		echo $html;
	}

	public function import_menu_capability_callback() {

		$html  = '<input autocomplete="off" type="text" id="daln-import-menu-capability" name="daln_import_menu_capability" class="regular-text" value="' . esc_attr( get_option( 'daln_import_menu_capability' ) ) . '" />';
		$html .= '<div class="help-icon" title="' . esc_attr__( 'The capability required to get access on the "Import" menu.', 'daln' ) . '"></div>';

		echo $html;
	}

	public function export_menu_capability_callback( $args ) {

		$html  = '<input autocomplete="off" type="text" id="daln-export-menu-capability" name="daln_export_menu_capability" class="regular-text" value="' . esc_attr( get_option( 'daln_export_menu_capability' ) ) . '" />';
		$html .= '<div class="help-icon" title="' . esc_attr__( 'The capability required to get access on the "Export" menu.', 'daln' ) . '"></div>';

		echo $html;
	}

	public function maintenance_menu_capability_callback() {

		$html  = '<input autocomplete="off" type="text" id="daln-maintenance-menu-capability" name="daln_maintenance_menu_capability" class="regular-text" value="' . esc_attr( get_option( 'daln_maintenance_menu_capability' ) ) . '" />';
		$html .= '<div class="help-icon" title="' . esc_attr__( 'The capability required to get access on the "Maintenance" menu.', 'daln' ) . '"></div>';

		echo $html;
	}

	/**
	 * The click on the "Export" button available in the "Export" menu is intercepted and the method that generates the
	 * downloadable XML file is called.
	 *
	 * @return void
	 */
	public function export_xml_controller() {

		/*
		 * Intercept requests that come from the "Export" button of the "Export" menu and generate the downloadable XML
		 * file.
		 */
		if ( isset( $_POST['daln_export'] ) ) {

			// Verify capability.
			if ( ! current_user_can( 'manage_options' ) ) {
				wp_die( esc_html__( 'You do not have sufficient permissions to access this page.' ) );
			}

			// Generate the header of the XML file.
			header( 'Content-Encoding: UTF-8' );
			header( 'Content-type: text/xml; charset=UTF-8' );
			header( 'Content-Disposition: attachment; filename=live-news-' . time() . '.xml' );
			header( 'Pragma: no-cache' );
			header( 'Expires: 0' );

			// Generate initial part of the XML file.
			$out  = '<?xml version="1.0" encoding="UTF-8" ?>';
			$out .= '<root>';

			// Generate the XML of the various db tables
			$out .= $this->shared->convert_db_table_to_xml( 'tickers', 'id' );
			$out .= $this->shared->convert_db_table_to_xml( 'featured_news', 'id' );
			$out .= $this->shared->convert_db_table_to_xml( 'sliding_news', 'id' );

			// Generate the final part of the XML file.
			$out .= '</root>';

			echo $out;
			die();

		}
	}

	/**
	 * Echo all the dismissible notices based on the values of the $notices array.
	 *
	 * @param array $notices An array of notices.
	 */
	public function dismissible_notice( $notices ) {

		foreach ( $notices as $key => $notice ) {
			echo '<div class="' . esc_attr( $notice['class'] ) . ' settings-error notice is-dismissible below-h2"><p>' . esc_html( $notice['message'] ) . '</p></div>';
		}
	}
}
